/** @preserve Eloquence WEBDLG v2 Plugin Example: Filetransfer
 (C) Copyright Marxmeier Software AG, 2024-2025
 $Id: webdlg-filetransfer.js,v 29.5 2025/10/21 14:17:07 rhg Exp $
*/
eq.plugin("Filetransfer", {
   // Attribute descriptor.
   atr : [
      [ 'buffer',    eq.atrGET|eq.atrSET|eq.atrINT|eq.atrNOC ],
      [ 'select',    eq.atrGET|eq.atrSET|eq.atrSTR
                                        |eq.atrLST|eq.atrNOC|eq.atrSYN ],
      [ 'upload',    eq.atrGET          |eq.atrSTR|eq.atrNOC|eq.atrSYN ],
      [ 'download',  eq.atrGET          |eq.atrSTR|eq.atrNOC|eq.atrSYN ],
      [ 'transfer',  eq.atrGET|eq.atrSET|eq.atrSTR
                                        |eq.atrLST|eq.atrNOC|eq.atrSYN ],
      [ 'close',               eq.atrSET|eq.atrINT|eq.atrNOC|eq.atrSYN ]
   ],

   // Class properties.
   prp : {
      minVersion  : 3,
      classList   : 'filetransfer',
      changeEvent : 'change',
      focusable   : false
   },

   // API: Constructor.
   apiInit : function(dlg, el) {
      var cls = this.constructor;
      if (cls.defaultFileName === undefined) {
         // First instance, setup default file name/type.
         cls.defaultFileName = eq.strings.get('filetransfer.file.name');
         if (!cls.defaultFileName)
            cls.defaultFileName = 'webdlg-download.dat';
         cls.defaultFileType = eq.strings.get('filetransfer.file.type');
         if (!cls.defaultFileType)
            cls.defaultFileType = 'application/octet-stream';
      }
   },

   // API: Set attribute value.
   apiSetAttr : function(dlg, id, el, atr, idx, val) {
      var cls = this.constructor;
      switch (atr) {
         case 'buffer':
            cls.bufferSize = val.i;
            break;
         case 'select':
            cls.setOptions(val.s);
            break;
         case 'transfer':
            cls.transfer.setNext(val.s);
            break;
         case 'close':
            if (val.i)
               cls.transfer.done();
            else
               cls.transfer.close();
      }
   },

   // API: Get attribute value.
   apiGetAttr : function(dlg, id, el, atr, idx) {
      var cls = this.constructor;
      switch (atr) {
         case 'buffer':
            if (cls.bufferSize <= 0)
               cls.bufferSize = 32766;
            return cls.bufferSize;
         case 'select':
            return cls.getOptions();
         case 'upload':
            return cls.transfer.start(
               new cls.upload(this, dlg, id, el));
         case 'download':
            return cls.transfer.start(
               new cls.download(this, el));
         case 'transfer':
            return cls.transfer.getNext();
      }
      return undefined;
   },

   static : {

      // Static API: Class installation.
      install : function() {
         this.bufferSize = 0;

         this.byteToHex = [];
         for (var i = 0; i < 256; i++)
            this.byteToHex.push(
               i.toString(16).padStart(2, '0'));
      },

      // Static API: Dispose static class properties.
      dispose : function() {
         this.transfer.close();
         if (this.options) {
            this.cleanupFilters();
            this.options.clear();
            this.options = null;
         }
      },

      // Implementation.
      // ---------------

      // File transfer object.
      transfer : {
         start : function(op) {
            this.close();
            return (this.op = op).start(this);
         },

         startTimer : function() {
            if (this.timeout)
               clearTimeout(this.timeout);
            this.timeout = setTimeout(this.close, 10000);
         },

         setNext : function(data) {
            if (this.op && this.op.setNext)
               return this.op.setNext(data);
         },

         getNext : function() {
            if (this.op && this.op.getNext)
               return this.op.getNext();
            return undefined;
         },

         done : function() {
            if (this.op && this.op.done)
               this.op.done();
            this.close();
         },

         close : function() {
            if (this.op) {
               this.op.dispose();
               this.op = undefined;
            }
            if (this.timeout) {
               clearTimeout(this.timeout);
               this.timeout = undefined;
            }
         }
      },

      // File upload class.
      upload : function(obj, dlg, id, el) {
         this.transfer = null;
         this.obj = obj;
         this.dlg = dlg;
         this.id = id;
         this.el = el;
         this.pending = true;

         this.start = function(transfer) {
            var
               self = this,
               cls = this.obj.constructor,
               sel = document.createElement('INPUT');

            if (cls.bufferSize <= 0)
               cls.bufferSize = 8000;

            sel.type = 'file';
            cls.setFileSelectorFilters(sel);
            sel.style.display = 'none';
            this.el.appendChild(sel);

            sel.onchange = function() {
               if (sel.files.length !== 1) {
                  transfer.close();
                  return;
               }

               // Obtain selected file,
               // purge file selector (no longer needed).
               var file = sel.files[0];
               self.el.textContent = ''; // Delete plugin child elements.
               self.el = undefined;

               // Return upload result.
               self.pending = false;
               self.obj.onUpdateReturn(self.dlg, self.id,
                  'file=' + file.name +
                  '\ntype=' + file.type +
                  '\nsize=' + file.size + '\n');

               // Start reading file.
               self.rd = new FileReader();

               self.rd.onloadend = function() {
                  if (self.rd.result) {
                     // File read successful.
                     self.data = new Uint8Array(self.rd.result);
                     self.rd = undefined;
                     self.offs = 0;
                     (self.transfer = transfer).startTimer();
                     if (self.pending) {
                        self.pending = false;
                        self.obj.onUpdateReturn(
                           self.dlg, self.id, self.returnData());
                     }
                  }
                  else {
                     // File read failed.
                     self.rd = undefined;
                     transfer.close();
                  }
               };

               self.rd.readAsArrayBuffer(file);
            };

            sel.oncancel = function() {
               transfer.close();
            };

            sel.click(); // Open file selector.
            return null; // Prepare asynchronous return.
         };

         this.returnData = function() {
            if (this.data && this.offs < this.data.length) {
               var
                  cls = this.obj.constructor,
                  toHex = cls.byteToHex,
                  part = this.data.slice(this.offs,
                                         this.offs + cls.bufferSize / 2),
                  l = part.length,
                  hex = [];
               this.offs += l;
               hex.length = l;
               for (var i = 0; i < l; i++)
                  hex[i] = toHex[part[i]];
               return hex.join('');
            }
            this.data = undefined;
            return undefined;
         };

         this.getNext = function() {
            if (this.offs !== undefined) {
               this.transfer.startTimer();
               return this.returnData();
            }
            // Data not available yet, prepare asynchronous return.
            this.pending = true;
            return null;
         };

         this.dispose = function() {
            if (this.pending) {
               this.pending = false;
               try {
                  this.obj.onUpdateReturn(this.dlg, this.id, undefined);
               } catch (e) {}
            }
            if (this.el) {
               this.el.textContent = ''; // Delete plugin child elements.
               this.el = undefined;
            }
            if (this.rd) {
               this.rd.abort();
               this.rd = undefined;
            }
            this.transfer = undefined;
            this.obj = undefined;
            this.dlg = undefined;
            this.id = undefined;
            this.data = undefined;
            this.offs = undefined;
         };
      },

      // File download class.
      download : function(obj, el) {
         this.transfer = null;
         this.obj = obj;
         this.el = el;
         this.data = [];

         this.start = function(transfer) {
            var
               cls = this.obj.constructor,
               fileType, fileName;
            if (cls.options) {
               fileType = cls.options.get('type');
               fileName = cls.options.get('file');
            }
            if (!fileType)
               fileType = cls.defaultFileType;
            if (!fileName)
               fileName = cls.defaultFileName;
            (this.transfer = transfer).startTimer();
            return 'file=' + fileName +
                   '\ntype=' + fileType + '\n';
         };

         this.setNext = function(data) {
            var l = data.length;
            if (l % 2) {
               console.error(
                  "BUG: Filetransfer.download.setNext odd data length");
               this.transfer.close();
               return;
            }
            for (var i = 0; i < l; i += 2)
               this.data.push(parseInt(data.substring(i, i + 2), 16));
            this.transfer.startTimer();
         };

         this.done = function() {
            if (this.data.length > 0) {
               var
                  cls = this.obj.constructor, 
                  url, lnk, fileType, fileName;

               if (cls.options) {
                  fileType = cls.options.get('type');
                  fileName = cls.options.get('file');
               }
               if (!fileType)
                  fileType = cls.defaultFileType;
               if (!fileName)
                  fileName = cls.defaultFileName;

               // eslint-disable-next-line
               url = (window.URL||window.webkitURL).createObjectURL(
                  new Blob([ new Uint8Array(this.data).buffer ],
                           { type : fileType }));
               this.data.length = 0;
               this.data = undefined;

               lnk = document.createElement('A');
               lnk.href = url;
               lnk.download = fileName;
               lnk.style.display = 'none';
               this.el.appendChild(lnk);

               lnk.click(); // Start download.

               // eslint-disable-next-line
               (window.URL||window.webkitURL).revokeObjectURL(url);
            }
         };

         this.dispose = function() {
            if (this.el) {
               this.el.textContent = ''; // Delete plugin child elements.
               this.el = undefined;
            }
            if (this.data) {
               this.data.length = 0;
               this.data = undefined;
            }
            this.transfer = undefined;
            this.obj = undefined;
         };
      },

      // Options.
      setOptions : function(s) {
         if (!this.options)
            this.options = new Map();
         var
            filter = null,
            opt = s.split('\n'),
            tok, tok_l, key, flt, flt_l;
         for (var opt_i = 0, opt_l = opt.length; opt_i < opt_l; opt_i++)
            if (tok_l = (tok = opt[opt_i].split('=')).length) {
               if ((key = tok[0]) === 'filter') {
                  if (!filter)
                     this.cleanupFilters();
                  if (   tok_l === 2
                      && (flt_l = (flt = tok[1].split('|')).length)
                      && (key = flt[0]).length) {
                        if (flt_l === 1) {
                           if (filter)
                              filter.delete(key);
                        }
                        else {
                           if (!filter)
                              this.options.set('filter', filter = new Map());
                           filter.set(key, flt.slice(1));
                        }
                     }
               }
               else if (key.length) {
                  if (tok_l === 2)
                     this.options.set(key, tok[1]);
                  else
                     this.options.delete(key);
               }
            }
      },
      getOptions : function() {
         if (this.options) {
            var arg = { self : this, s : '' };
            this.options.forEach(this.getOption, arg);
            return arg.s;
         }
         return '';
      },
      getOption : function(val, key) {
         if (key === 'filter')
            val.forEach(this.self.getFilterOption, this);
         else
            this.s += key + '=' + val + '\n';
      },
      getFilterOption : function(val, key) {
         this.s += 'filter=' + key + '|' + val.join('|') + '\n';
      },

      // Filters.
      setFileSelectorFilters : function(sel) {
         var filter, ext;
         if (this.options && (filter = this.options.get('filter'))) {
            ext = [];
            filter.forEach(this.getFilterExt, ext);
            if (ext.length)
               sel.accept = '.' + ext.join(',.');
         }
      },
      getFilterExt : function(val) {
         for (var i = 0, l = val.length; i < l; i++) {
            var ext = val[i];
            if (ext.length)
               this.push(ext);
         }
      },
      cleanupFilters : function() {
         var filter = this.options.get('filter');
         if (filter) {
            filter.forEach(this.cleanupFilter);
            filter.clear();
            this.options.delete('filter');
         }
      },
      cleanupFilter : function(val) {
         val.length = 0;
      }
   }
});
